package ga.view.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * This is an extended {@link ArrayList} that parses and holds all
 * {@link GASettings}.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("serial")
public class ViewSettings extends ArrayList<GASettings> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(ViewSettings.class
      .getName());

  /**
   * Instantiates a new view settings.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ViewSettings() {
  }

  /**
   * Instantiates a new view settings and parses the {@link GASettings} from
   * xml.
   * 
   * @param e
   *          The xml Element
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ViewSettings(final Element e) {
    parse(e);
  }

  /**
   * Builds the viewsettings from xml.
   * 
   * @param filename
   *          the filename to load
   * @return the view settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static ViewSettings buildFromXml(final String filename) {
    return buildFromXml(ViewSettings.class.getClassLoader()
        .getResourceAsStream(filename));
  }

  /**
   * Builds the viewsettings from xml.
   * 
   * @param file
   *          the file to load
   * @return the view settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static ViewSettings buildFromXml(final File file) {
    try {
      return buildFromXml(new FileInputStream(file));
    } catch (final FileNotFoundException e) {
      e.printStackTrace();
      return null;
    }
  }

  /**
   * Builds the from xml.
   * 
   * @param is
   *          the input stream to load
   * @return the view settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static ViewSettings buildFromXml(final InputStream is) {
    final SAXBuilder builder = new SAXBuilder(false);
    try {
      final Element e = builder.build(is).getRootElement();

      if (e == null) {
        throw new RuntimeException(
            "Error parsing settings, application will not work");
      }

      return new ViewSettings(e);
    } catch (final JDOMException e) {
      LOGGER.log(Level.SEVERE, "Failed to parse file", e);
    } catch (final IOException e) {
      LOGGER.log(Level.SEVERE, "Failed to load file", e);
    } finally {
      try {
        is.close();
      } catch (final Exception e) {
      }
    }

    return null;
  }

  /**
   * Parses the xml elements and creates all objects.
   * 
   * @param e
   *          the xml Element.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void parse(final Element e) {
    final Iterator<?> it = e.getChildren("setting", e.getNamespace())
        .iterator();
    while (it.hasNext()) {
      final Element iga = (Element) it.next();
      add(new GASettings(iga));
    }
  }

  /**
   * Gets a {@link GASettings} instance by its name.
   * 
   * @param name
   *          Name of the settings.
   * @return the gA settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public GASettings get(final String name) {
    for (final GASettings setting : this) {
      if (setting.get("name").equals(name)) {
        return setting;
      }
    }
    return null;
  }

  /**
   * The main method for testing.
   * 
   * @param args
   *          the arguments
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final ViewSettings vs = ViewSettings
        .buildFromXml("example/SimpleObjectGA.xml");
    LOGGER.info(vs.toString());
  }
}
